/**
 * jobqueue.c
 *
 * Дана програма моделює роботу багатопотокового паралельного сервера без зворотного зв'язку
 * між серверними і клієнтським потоками (але якщо клієнтський потік представляє зовнішніх
 * клієнтів, може мати місце зворотний зв'язок між серверними потоками і зовнішніми клієнтами;
 * для цього клієнтський потік в структуру, яка описує завдання для серверного потоку, повинен
 * включати інформацію про спосіб зв'язку з зовнішнім клієнтом) і кешем результатів, який 
 * спільно використовуєтсья серверними потоками. Використовується фіксована
 * кількість завчасно створених серверних потоків. Один клієнтський потік 
 * (який, можливо, представляє зовнішніх клієнтів) періодично виробляє завдання (jobs) для
 * серверних потоків, які розміщує в черзі завдань (jobqueue). Серверні потоки витягують
 * завдання з черги завдань і обробляють їх. Клієнтський потік, помістивши завдання
 * в чергу, продовжує роботу (наприклад, чекає запиту від нового зовнішнього клієнта).
 * Серверний потік, обробляючи завдання, спочатку шукає результат в кеші, і виконує власне
 * обробку, тільки якщо в кеші не виявиться відповідного результату. Отимавши новий результат,
 * серверний потік поміщає його в кеш. Кеш реалізовано як зв'язаний список структур. 
 * Доступ серверних потоків до нього регулюється за допомогою м'ютекса.
 * Підтримується обмеження на розмір кешу. Дані поміщаютсья в кеш за дисципліною
 * "першим зайшов - першим вийшов".
 * Програма виводить повідомлення про розміщення в черзі чергового завдання і результат
 * обробки кожного завдання.
 * Черга завдань реалізована як зв'язаний список структур. Доступ до неї регулюється
 * за допомогою  м'ютекса (job_queue_mutex) і умовної змінної (job_queue_cond). 
 */
 
#include <assert.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* Кількість серверних потоків */
enum { NSERVERS = 5 };
/* Обмеження на розмір кешу результатів (у байтах) */
enum { RESULT_CACHE_MAXSIZE = 50 };
/* Максимальне значення ідентифікатора завдання */
enum { JOB_MAXID = 9 };
/* Структура, яка представляє завдання */
struct job 
{
        int id;                 /* Ідентифікатор завдання 
                                   (ідентифікує також і результат) */
        /* ... */
        struct job *next;
};

/* Покажчики на початок та кінець черги завдань */
struct job *job_queue_head, *job_queue_tail;

pthread_mutex_t job_queue_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t job_queue_cond;

/* Структура, яка представляє результат обробки завдання */
struct result
{
        int id;                 /* Ідентифікатор результату (є також
                                   ідентифікатором відповідного
                                   завдання) */
        /* ... */
        struct result *next;    
};
/* Покажчики на початок та кінець кешу результатів */
struct result *result_cache_head, *result_cache_tail;
/* Розмір кешу */
int result_cache_size;

pthread_mutex_t result_cache_mutex = PTHREAD_MUTEX_INITIALIZER;

int init_job_queue();
int init_result_cache();
void client(void), *server(void *);


int main(int argc, char **argv)
{
        pthread_t server_threads[NSERVERS];
        int server_args[NSERVERS];
        int i, rval, terrno;

        /* Виконує ініціалізацію черги завдань. */
        if (init_job_queue() != 0)
                exit(EXIT_FAILURE);
        /* Виконує ініціалізацію кешу результатів. */
        if (init_result_cache() != 0)
                exit(EXIT_FAILURE);
        /* Створює серверні потоки. */
        for (i = 0; i < NSERVERS; i++) {
                server_args[i] = i;
                terrno = pthread_create(&server_threads[i], NULL, server,
                                                        &server_args[i]);
                if (terrno != 0) {
                        fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(terrno));
                        exit(EXIT_FAILURE);
                }
        }

        /* Заходить в основний робочий цикл. */
        client();

        /* Чекає завершення серверних потоків. */
        for (i = 0; i < NSERVERS; i++) {
                rval = pthread_join(server_threads[i], NULL);
                assert(rval == 0);
        }	

        exit(EXIT_SUCCESS);
}

/**
 * Розміщує структуру для нового завдання і виконує її ініціалізацію.
 * Повертає: покажчик на створену структуру у випадку успіху,
 * NULL - у випадку невдачі.
 */
struct job* create_job(int id)
{
        struct job *new_job;

        new_job = malloc(sizeof(*new_job));
        if (new_job == NULL) {
                fprintf(stderr, "Not enough memory to create new job\n");
                return NULL;
        }
        new_job->id = id;
        /* ... */
        new_job->next = NULL;
        return new_job;
}

/**
 * Звільняє пам'ять, виділену для завдання.
 */
void free_job(struct job *job)
{
        assert(job != NULL);
        free(job);
}

/**
 * Виконує ініціалізацію черги завдань.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int init_job_queue()
{
        int rval; 

        job_queue_head = NULL;
        job_queue_tail = NULL;
        rval = pthread_cond_init(&job_queue_cond, NULL);
        if (rval != 0) {
                fprintf(stderr, "Error initializing conditional variable:"
                                                " %s\n", strerror(rval));
                return 1;
        }
        return 0;
}

/**
 * Додає завдання до (хвоста) черги завдань.
 * Аргументи: new_job - покажчик на структуру із новим завданням.
 */
void enqueue_job(struct job *new_job)
{
        int rval;

        assert(new_job != NULL);

        rval = pthread_mutex_lock(&job_queue_mutex);
        assert(rval == 0);

        if (job_queue_tail == NULL) {
                assert(job_queue_head == NULL);
                job_queue_head = new_job;
        } else
                job_queue_tail->next = new_job;
        job_queue_tail = new_job;
        assert(job_queue_head != NULL);

        printf("Job %d has been requested\n", new_job->id);

        /* Будить один із серверних потоків, що, можливо, чекають появи
           в черзі нового завдання. */
        rval = pthread_cond_signal(&job_queue_cond);
        assert(rval == 0);

        rval = pthread_mutex_unlock(&job_queue_mutex);
        assert(rval == 0);

        return;
}

/**
 * Вилучає завдання з (голови) черги завдань.
 * Повертає покажчик на структуру з вилученим із черги завданням.
 */
struct job* dequeue_job()
{
        struct job *next_job;
        int rval;
    
        rval = pthread_mutex_lock(&job_queue_mutex);
        assert(rval == 0);

        /* Якщо черга порожня, чекає, доки в ній з'явиться завдання. */
        while (job_queue_head == NULL)
                rval = pthread_cond_wait(&job_queue_cond,
                                                &job_queue_mutex);
        assert(rval == 0);

        next_job = job_queue_head;
        job_queue_head = job_queue_head->next;
        if (job_queue_head == NULL)
                job_queue_tail = NULL;

        rval = pthread_mutex_unlock(&job_queue_mutex);
        assert(rval == 0);

        return next_job;
}

/**
 * Основний робчий цикл клієнтського потоку.
 */
void client()
{
        for (;;) {
                struct job *new_job;

                /* Виконує якусь роботу (можливо, взаємодіє з зовнішнім
                   клієнтом)... */

                /* Створює нове завдання. */
                new_job = create_job(random() % (JOB_MAXID + 1));
                if (new_job == NULL)
                        continue;
                /* Додає завдання до черги. */
                enqueue_job(new_job);

                /* Виконує якусь іншу роботу... */

                /* Це тут тільки для того, щоб уповільнити роботу
                   клієнтського потоку. */
                sleep(1); 
        }
        return;
}

/**
 * Розміщує структуру для нового результату обробки завдання і виконує
 * її ініціалізацію.
 * Повертає: покажчик на створену структуру у випадку успіху,
 * NULL - у випадку невдачі.
 */
struct result *create_result(int id)
{
        struct result *new_result;

        new_result = malloc(sizeof(*new_result));
        if (new_result == NULL) {
                fprintf(stderr, "Not enough memory to create new"
                                                        " result\n");
                return NULL;
        }
        new_result->id = id;
        /* ... */
        new_result->next = NULL;
        return new_result;
}

/**
 * Створює дублікат результату обробки завдання, на який указує result.
 * Повертає: покажчик на створений дублікат у випадку успіху,
 * NULL - у випадку невдачі.
 */
struct result *dup_result(struct result *result)
{
        struct result *dup_result;

        assert(result != NULL);

        dup_result = malloc(sizeof(*dup_result));
        if (dup_result == NULL) {
                fprintf(stderr, "Not enough memory to create new"
                                                        " result\n");
                return NULL;
        }
        dup_result->id = result->id;
        /* ... */
        dup_result->next = NULL;
        return dup_result;
}

/**
 * Повертає розмір результату обробки завдання, на який указує result.
 */
int get_result_size(struct result *result)
{
        return sizeof(struct result);
}

/**
 * Звільняє пам'ять, виділену для результату обробки завдання.
 */
void free_result(struct result *result)
{
        assert(result != NULL);
        free(result);
}

/**
 * Виконує ініціалізацію кешу результатів.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int init_result_cache()
{
        result_cache_head = NULL;
        result_cache_tail = NULL;
        result_cache_size = 0;
        return 0;
}

/**
 * Шукає заданий ідентифікатором id результат у кеші результатів.
 * Повертає: покажчик на дублікат результату у випадку успіху,
 * NULL у випадку невдачі.
 */
struct result *find_result_in_cache(int id)
{
        struct result *result;
        int rval;

        /* Блокує доступ до кешу результатів для інших потоків. */
        rval = pthread_mutex_lock(&result_cache_mutex);
        assert(rval == 0);
        for (result = result_cache_head;
                                result != NULL; 
                                        result = result->next) {
                if (result->id == id) {
                        /* Заданий результат у кеші знайдено.
                           Створює дублікат знайденого результату. */
                        result = dup_result(result);
                        break;
                }
        }
        /* Розблоковує кеш результатів. */
        rval = pthread_mutex_unlock(&result_cache_mutex);
        assert(rval == 0);
        return result;
}

/**
 * Зберігає дублікат результату обробки завдання, на який указує result, 
 * у кеші результатів.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int save_result_in_cache(struct result *new_result)
{
        struct result *result;
        int new_result_size;
        int rval;

        assert(result != NULL);

        /* Забезпечує виконання обмеження на розмір кешу. */
        new_result_size = get_result_size(new_result);
        if (new_result_size > RESULT_CACHE_MAXSIZE) {
                fprintf(stderr, "Result for job %d is too big to cache\n",
                                                        new_result->id);
                return 1;
        }
        /* Блокує доступ до кешу результатів для інших потоків. */
        rval = pthread_mutex_lock(&result_cache_mutex);
        assert(rval == 0);
        /* Перевіряє, чи немає в кеші вже такого результату. */
        for (result = result_cache_head; 
                                result != NULL; 
                                        result = result->next) {
                if (result->id == new_result->id) {
                        /* Такий результат у кеші вже є.
                           Розблоковує кеш результатів. */
                        rval = pthread_mutex_unlock(&result_cache_mutex);
                        assert(rval == 0);
                        fprintf(stderr, "Result for job %d is in cache"
                                        " yet\n", new_result->id);
                        return 0;
                }
        }
        /* Такого результату в кеші ще немає.
           Забезпечує виконання обмеження на розмір кешу. */
        while (result_cache_size + new_result_size > 
                                        RESULT_CACHE_MAXSIZE) {
                assert(result_cache_head != NULL);
                result = result_cache_head;
                result_cache_head = result_cache_head->next;
                if (result_cache_head == NULL)
                        result_cache_tail = NULL;
                result_cache_size -= get_result_size(result);
                fprintf(stderr, "Result for job %d has been removed from"
                                                " cache\n", result->id);
                free_result(result);
        }
        /* Створює дублікат нового результату. */
        result = dup_result(new_result);
        if (result == NULL) {
                /* Розблоковує кеш результатів. */
                rval = pthread_mutex_unlock(&result_cache_mutex);
                assert(rval == 0);
                fprintf(stderr, "Not enough memory to cache result for"
                                        " job %d\n", new_result->id);
                return 1;
        }
        /* Додає дублікат нового результату до кешу. */
        if (result_cache_tail == NULL) {
                assert(result_cache_head == NULL);
                result_cache_head = result;
        } else
                result_cache_tail->next = result;
        result_cache_tail = result;
        result_cache_size += new_result_size;
        /* Розблоковує кеш результатів. */
        rval = pthread_mutex_unlock(&result_cache_mutex);
        assert(rval == 0);
        fprintf(stderr, "Result for job %d has been saved in cache\n",
                                                        new_result->id);
        return 0;
}

/**
 * Виконує обробку завдання.
 * Аргументи: job - покажчик на структуру з завданням, яке треба обробити.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int process_job(struct job *job)
{
        struct result *result;

        assert(job != NULL);

        /* Шукає результат у кеші. */
        result = find_result_in_cache(job->id);
        if (result != NULL) {
                /* Знайшла. */
                printf("Result for job %d has been found in cache\n",
                                                                job->id);
        } else {
                /* Не знайшла, виконує обробку самостійно. */
                printf("Result for job %d has not been found in cache\n",
                                                                job->id);
                /* ... */
                
                /* Розміщує результат у структурі типу result. */
                result = create_result(job->id);
                if (result == NULL)
                        return 1;
                /* Зберігає результат у кеші (пробує, принаймні). */
                save_result_in_cache(result);
        }
        /* Виконує якісь дії (наприклад, повідомляє результат зовнішньому
           клієнту, записує повідомлення в журнал)... */
        free_result(result);
        return 0;
}

/**
 * Головна функція серверного потоку.
 * Аргументи: arg - покажчик на змінну типу int, яка зберігає номер
 * серверного потоку.
 */
void *server(void *arg)
{
        int number;             /* Номер серверного потоку */

        number = *((int *) arg);

        for (;;) {
                struct job *new_job;

                /* Вилучає з черги нове завдання. */
                new_job = dequeue_job();
                /* Обробляє завдання. */
                if (process_job(new_job) != 0)
                        fprintf(stderr, "Error processing job %d by"
                                " server %d\n", new_job->id, number);
                else
                        printf("Job %d has been successfully processed"
                                " by server %d\n", new_job->id, number);
                /* Звільняє пам'ять, виділену для завдання. */
                free_job(new_job);
        }
        return NULL;
}
